Import "TVector2f.bmx"
Import "TParticle.bmx"
Import "TParticleAttractor.bmx"
Import "TParticleEmitterProps.bmx"
Include "TParticleEmitter.bmx"

Rem

	Particle system

End Rem
Type TParticleSystem

	' List of active particles
	Field particles:TList
	' List of inactive particles
	Field particlePool:TList
	' List of attractors
	Field attractors:TList
	' List of emitters
	Field emitters:TList
	
	' Particle system bounds
	Field minBounds:TVector2f
	Field maxBounds:TVector2f
	
	' Number of active particles in the system
	Field particleCount:Int
	' Maximum number of particles in the system
	Field maxParticles:Int
	' Number of particles currently in the pool
	Field poolSize:Int
	
	' Whether the attraction is on
	Field isAttracting:Int
	
	' Particle texture for the moment we'll just store it in the system until
	' we move it into the emitters
	Field textureImage:TPixmap
	Field texturePointer:Int
	Field textureU:Float
	Field textureV:Float
	
	Field particleSize:Float = 4.0
	Field zDistance:Float = -10.0
	Field friction:Float = 1.0
	
	Field color:Float[] = [0.2, 0.4, 1.0, 0.4]
	
	Field particleDisplayDelay = 10
	
	Rem
		Particle system constructor
	End Rem
	Method New()
		Self.particleCount = 0
		Self.particles = CreateList()
		Self.particlePool = CreateList()
		Self.attractors = CreateList()
		Self.emitters = CreateList()
		Self.minBounds = New TVector2f
		Self.maxBounds = New TVector2f
		Self.maxParticles = 0
		Self.poolSize = 0
		Self.isAttracting = True
	End Method
	
	Rem
		Initialise the particle system
	End Rem
	Method initSystem(maxParticles:Int, minBounds:TVector2f, maxBounds:TVector2f)
		Self.maxParticles = maxParticles
		Self.minBounds.x = minBounds.x
		Self.minBounds.y = minBounds.y
		Self.maxBounds.x = maxBounds.x
		Self.maxBounds.y = maxBounds.y
		
		' Populate the particle pool
		For Local i:Int = 1 To maxParticles
			Local p:TParticle = New TParticle
			ListAddLast Self.particlePool, p
		Next
		Self.poolSize = maxParticles
		
	End Method
	
	Rem
		Specify the texture to be used by the particle system
	End Rem
	Method setTexture(texturePath:String)		
		Self.textureImage = LoadPixmap(texturePath)
		DebugLog("Loaded texture with dimensions: (" + Self.textureImage.width + ", " + Self.textureImage.height + ")")
		Self.texturePointer = GLTexFromPixmap(Self.textureImage)
		GLAdjustTexSize(Self.textureImage.width, Self.textureImage.height)
		
		Self.textureU = 1.0
		Self.textureV = 1.0
	End Method
	
	Rem
		Count the number of particles in the pool
	End Rem
	Method countAvailableParticles()
		Return Self.poolSize
	End Method

	Rem
		Emit a particle
	End Rem
	Method emitParticle(position:TVector2f, angle:Float, velocity:Float, energy:Int, colour:Float[])
		If Self.poolSize > 0 Then
			Local p:TParticle = TParticle(Self.particlePool.RemoveLast())
			p.initParticle(position, angle, velocity, energy, colour)
			ListAddLast Self.particles, p
			Self.particleCount :+ 1
			Self.poolSize :- 1
		End If
	End Method
	
	Rem
		Add an emitter to the system
	End Rem
	Method addEmitter(emitter:TParticleEmitter)
		ListAddLast Self.emitters, emitter
	End Method
	
	Rem
		Add an attractor to the system
	End Rem
	Method addAttractor(attractor:TParticleAttractor)
		ListAddLast Self.attractors, attractor
	End Method

	Rem
		Update the particle system
	End Rem
	Method tick()
	
		' Update all the emitters in the system
		For Local emitter:TParticleEmitter = EachIn Self.emitters
			emitter.tick()
		Next
	
		' Update all the particles in the system
		Local pLink:TLink = Self.particles.FirstLink()	
		While pLink
		
			Local particle:TParticle = TParticle(pLink.Value())
		
			If particle.energy = 0 Then
				' Return the particle to the pool
				pLink.Remove()
				ListAddLast Self.particlePool, particle
				Self.particleCount :- 1
				Self.poolSize :+ 1
			Else
				
				If Self.isAttracting Then
					For Local attractor:TParticleAttractor = EachIn Self.attractors
						attractor.attract(particle)
					Next
				End If
				
				particle.v.multiply(Self.friction)
				
				particle.tick()
				
			End If
			
			pLink = pLink.NextLink()
		
		Wend
		
	End Method

	Method render()
		Local vertices:Float[]
		Local texCoords:Float[]
		Local colours:Float[]
		
		vertices = New Float[Self.particleCount * 12] ' 4 * 3D vertices per particle
		texCoords = New Float[Self.particleCount * 8]
		colours = New Float[Self.particleCount * 16]  ' RGBA colours

		Local index:Int = 0
						
		For particle:TParticle = EachIn Self.particles
			
			Local r:Float = particle.colour[0]
			Local g:Float = particle.colour[1]
			Local b:Float = particle.colour[2]
			Local a:Float = Float(particle.energy) / Float(particle.startEnergy)
			Local particleSize:Float = Self.particleSize * a
			If particle.startEnergy - particle.energy < Self.particleDisplayDelay
				a = 0.0
			End If
			
			Local vertexOffset:Int = index * 12
			Local texOffset:Int = index * 8
			Local colourOffset:Int = index * 16
			' Add current particle to the vertex array
			vertices[vertexOffset + 0] = particle.p.x - particleSize  ' x1
			vertices[vertexOffset + 1] = particle.p.y - particleSize  ' y1
			vertices[vertexOffset + 2] = 0.0                               ' z1
			vertices[vertexOffset + 3] = particle.p.x - particleSize  ' x2
			vertices[vertexOffset + 4] = particle.p.y + particleSize  ' y2
			vertices[vertexOffset + 5] = 0.0                          ' z2
			vertices[vertexOffset + 6] = particle.p.x + particleSize  ' x3
			vertices[vertexOffset + 7] = particle.p.y + particleSize  ' y3
			vertices[vertexOffset + 8] = 0.0                          ' z3
			vertices[vertexOffset + 9] = particle.p.x + particleSize  ' x4
			vertices[vertexOffset + 10] = particle.p.y - particleSize ' y4
			vertices[vertexOffset + 11] = 0.0                              ' z4			' Add the current particle's texture coords
			texCoords[texOffset + 0] = 0                 ' u1
			texCoords[texOffset + 1] = 0                 ' v1
			texCoords[texOffset + 2] = 0                 ' u2
			texCoords[texOffset + 3] = Self.textureV     ' v2
			texCoords[texOffset + 4] = Self.textureU     ' u3
			texCoords[texOffset + 5] = Self.textureV     ' v3
			texCoords[texOffset + 6] = Self.textureU     ' u4
			texCoords[texOffset + 7] = 0                 ' v4
			
			'Rem
			' Set the particle colour
			colours[colourOffset + 0] = r
			colours[colourOffset + 1] = g
			colours[colourOffset + 2] = b
			colours[colourOffset + 3] = a
			colours[colourOffset + 4] = r
			colours[colourOffset + 5] = g
			colours[colourOffset + 6] = b
			colours[colourOffset + 7] = a
			colours[colourOffset + 8] = r
			colours[colourOffset + 9] = g
			colours[colourOffset + 10] = b
			colours[colourOffset + 11] = a
			colours[colourOffset + 12] = r
			colours[colourOffset + 13] = g
			colours[colourOffset + 14] = b
			colours[colourOffset + 15] = a
			'End Rem

			index :+ 1

		Next

		' Draw the particles
		'glEnable(GL_TEXTURE_2D)
		glBindTexture(GL_TEXTURE_2D, Self.texturePointer)
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR)
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR)

		glEnableClientState(GL_VERTEX_ARRAY)
		glEnableClientState(GL_TEXTURE_COORD_ARRAY)
		glEnableClientState(GL_COLOR_ARRAY)
		glPushMatrix()
			' 2D space floating point vertices and textures
			glVertexPointer(3, GL_FLOAT, 0, Float Ptr vertices)
			glTexCoordPointer(2, GL_FLOAT, 0, Float Ptr texCoords)
			glColorPointer(4, GL_FLOAT, 0, Float Ptr colours)
			glTranslatef(0.0, 0, Self.zDistance)
			'glColor4f(Self.color[0], Self.color[1], Self.color[2], Self.color[3])
			glDrawArrays(GL_QUADS, 0, Self.particleCount * 4)
		glPopMatrix()
		
		glDisableClientState(GL_VERTEX_ARRAY)
		glDisableClientState(GL_TEXTURE_COORD_ARRAY)
		glDisableClientState(GL_COLOR_ARRAY)
	End Method

End Type